/* $Id: io.c,v 1.72 1999/10/04 23:42:41 donwm Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Taken from E1431 library, heavily modified by Eric Backus */

/* Routines for low level I/O to E1432 module */

#include "sema.h"

/* #define	DEBUG_XFER */		/* For i1432_xfer_block */

#define	DONE_TIMEOUT		4.0
#define	AUTO_ZERO_TIMEOUT	8.0
#define	CMD_READY_TIMEOUT	1.0

/*#define	DOUBLE_READ_PRINT*/

#define	ERROR_LOOP_MAX		32

/* I/O intependent version of SICL's iintroff */
SHORTSIZ16
i1432_introff(void)
{
#ifdef	HAVE_SICL
    int ierror = iintroff();
    if (ierror != 0)
	return i1432_print_error(ERR1432_SICL_ERROR);
    return 0;
#endif
#ifdef	HAVE_VTL
    return i1432_visa_introff();
#endif
#ifdef	HPVXI_DOWNLOAD
    return 0;	/* not implemented */
#endif
}

/* I/O intependent version of SICL's iintron */
SHORTSIZ16
i1432_intron(void)
{
#ifdef	HAVE_SICL
    int ierror = iintron();
    if (ierror != 0)
	return i1432_print_error(ERR1432_SICL_ERROR);
    return 0;
#endif
#ifdef	HAVE_VTL
    return i1432_visa_intron();
#endif
#ifdef	HPVXI_DOWNLOAD
    return 0;	/* not implemented */
#endif
}

/*
 *********************************************************************
 This reports a bus error.
 *********************************************************************
 */
SHORTSIZ16
i1432_buserr_report(SHORTSIZ16 la, long addr, char *str)
{
    char error_info[64];

    (void) sprintf(error_info, "Bus error in %s at LA %d, register 0x%lx",
		   str, la, addr);
    i1432_error_info = error_info;
    return i1432_la_print_error(la, ERR1432_BUS_ERROR);
}

#ifdef	HAVE_SIGBUS
/*
 *********************************************************************
 This is the bus error handler to catch bus error due to read and
 writes to non-existant registers on the VXI bus.
 *********************************************************************
 */
/*ARGSUSED*/
void
i1432_buserr_handler(LONGSIZ32 sig, LONGSIZ32 code,
		     struct sigcontext *scp)
{
    if (i1432_buserr_env_valid)
    {
	if (sig == SIGBUS)
	    /* return to previous context, signal = 1 */
	    longjmp(i1432_buserr_env, 1);
	else
	    PRINTF(("E1432: Illegal signal value to"
		    "bus_error_handler():  Signal = %ld\n", sig));
    }
    else
	PRINTF(("The E1432 library caught an unexpected signal %ld.\n"
		"THIS SIGNAL PROBABLY DID NOT COME FROM THE E1432\n"
		"LIBRARY, but rather came from some OTHER part of\n"
		"the software.  Use the E1432 library function\n"
		"\"e1432_set_try_recover()\" to disable signal\n"
		"trapping\n", sig));
    exit(1);
}
#endif /* HAVE_SIGBUS */

/*
 *********************************************************************
 Waits until the specified bits <bitMask> of the register of the at
 la, reach the value specified in <bitState>.  The waiting for (status
 & bitMask) == bitState cannot exceed the time value specified in
 <timeout>.  Returns 0 on successfull completion, Otherwise returns an
 error if the timeout as been reached, and appends the error string
 pointed to by <errorStr> to the displayed message.
 la;           vxi address
 reg;          register
 bitMask;      mask for bits to be tested
 bitState;     bitValue to be tested
 timeout;      timeout for testing the status
 *errorStr;    additional error message
 *********************************************************************
 */
SHORTSIZ16
i1432_wait_bits_match(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 reg,
		      SHORTSIZ16 bit_mask, SHORTSIZ16 bit_state,
		      FLOATSIZ64 timeout, char *errorStr)
{
    SHORTSIZ16 error, bits;
    FLOATSIZ64 startTime, currentTime;

    bit_state &= bit_mask;

    i1432_get_time(&startTime);
    do
    {
	i1432_get_time(&currentTime);

	error = i1432_direct_read_register(mn, reg, &bits);
	if (error)
	    return error;
	if ((bits & bit_mask) == bit_state)
	    return 0;
    }
    while (currentTime - startTime < timeout);

    i1432_error_info = errorStr;
    return i1432_la_print_error(mn->la, ERR1432_STATUS_READ_TIMEOUT);
}

SHORTSIZ16
i1432_wait32_bits_match(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 reg,
			LONGSIZ32 bit_mask, LONGSIZ32 bit_state,
			FLOATSIZ64 timeout, char *errorStr)
{
    LONGSIZ32 bits;
    SHORTSIZ16 error;
    FLOATSIZ64 startTime, currentTime;

    bit_state &= bit_mask;

    i1432_get_time(&startTime);
    do
    {
	i1432_get_time(&currentTime);

	error = i1432_direct_read32_register(mn, reg, &bits);
	if (error)
	    return error;
	if ((bits & bit_mask) == bit_state)
	    return 0;
    }
    while (currentTime - startTime < timeout);

    i1432_error_info = errorStr;
    return i1432_la_print_error(mn->la, ERR1432_STATUS_READ_TIMEOUT);
}

SHORTSIZ16
i1432_error_wait32_bits_match(E1432_MODULE_LIST_NODE *mn,
			      LONGSIZ32 reg, LONGSIZ32 bit_mask,
			      LONGSIZ32 bit_state, FLOATSIZ64 timeout,
			      char *errorStr)
{
    LONGSIZ32 bits;
    SHORTSIZ16 error, bits16;
    FLOATSIZ64 startTime, currentTime;

    bit_state &= bit_mask;

    i1432_get_time(&startTime);
    do
    {
	i1432_get_time(&currentTime);

	error = i1432_direct_read_register(mn, E1432_STATUS_REG,
					   &bits16);
	if (error)
	    return error;
	if ((bits16 & E1432_STATUS_ERR_N) == 0)
	    return -1;	/* Caller must call i1432_error_check */

	error = i1432_direct_read32_register(mn, reg, &bits);
	if (error)
	    return error;
	if ((bits & bit_mask) == bit_state)
	    return 0;
    }
    while (currentTime - startTime < timeout);

    i1432_error_info = errorStr;
    return i1432_la_print_error(mn->la, ERR1432_STATUS_READ_TIMEOUT);
}

SHORTSIZ16
i1432_error_wait32_bits_no_match(E1432_MODULE_LIST_NODE *mn,
				 LONGSIZ32 reg, LONGSIZ32 bit_mask,
				 LONGSIZ32 bit_state,
				 FLOATSIZ64 timeout, char *errorStr)
{
    LONGSIZ32 bits;
    SHORTSIZ16 error, bits16;
    FLOATSIZ64 startTime, currentTime;

    bit_state &= bit_mask;

    i1432_get_time(&startTime);
    do
    {
	i1432_get_time(&currentTime);

	error = i1432_direct_read_register(mn, E1432_STATUS_REG,
					   &bits16);
	if (error)
	    return error;
	if ((bits16 & E1432_STATUS_ERR_N) == 0)
	    return -1;	/* Caller must call i1432_error_check */

	error = i1432_direct_read32_register(mn, reg, &bits);
	if (error)
	    return error;
	if ((bits & bit_mask) != bit_state)
	    return 0;
    }
    while (currentTime - startTime < timeout);

    i1432_error_info = errorStr;
    return i1432_la_print_error(mn->la, ERR1432_STATUS_READ_TIMEOUT);
}

SHORTSIZ16
i1432_error_check(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error2, status;
    LONGSIZ32 mod_error, prev_mod_error;
    int     count;

    error = i1432_get_module_from_chan(hw, ID, &mn);
    if (error)
	return error;

    error = i1432_introff();
    if (error)
	return error;

    /* Check ERROR */
    error = i1432_direct_read_register(mn, E1432_STATUS_REG,
				       &status);
    if (error < 0)
	goto cleanup;
    if ((status & E1432_STATUS_ERR_N) == 0)
    {
	count = 0;
	prev_mod_error = ERR1432_ERROR_QUEUE_CORRUPT;
	do
	{
	    error = i1432_write_cmd0(hw, ID, E1432_CMD_READ_ERROR);
	    if (error < 0)
		goto cleanup;
	    error = i1432_read_resp(hw, ID, &mod_error);
	    if (error < 0)
		goto cleanup;
	    if (mod_error < 0)
	    {
		(void) i1432_la_print_error(mn->la, (SHORTSIZ16) mod_error);
		prev_mod_error = mod_error;
	    }
	    count++;
	}
	while (mod_error < 0 && count < ERROR_LOOP_MAX);

	if (prev_mod_error == ERR1432_ERROR_QUEUE_CORRUPT ||
	    count == ERROR_LOOP_MAX)
	{
	    error = i1432_la_print_error(mn->la, ERR1432_ERROR_QUEUE_CORRUPT);
	    goto cleanup;
	}

	/* Sanity check */
	error = i1432_direct_read_register(mn, E1432_STATUS_REG,
					   &status);
	if (error < 0)
	    goto cleanup;
	if ((status & E1432_STATUS_ERR_N) == 0)
	{
	    error = i1432_la_print_error(mn->la, ERR1432_ERROR_QUEUE_CORRUPT);
	    goto cleanup;
	}

	error = (SHORTSIZ16) prev_mod_error;
	goto cleanup;
    }

 cleanup:
    error2 = i1432_intron();
    if (error2)
	return error2;

    if (error)
	return error;

    return 0;
}

SHORTSIZ16
i1432_read_resp(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 *resp)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;

    error = i1432_get_module_from_chan(hw, ID, &mn);
    if (error)
	return error;

    /* Wait for Q RESP READY */
    error = i1432_wait_bits_match(mn, E1432_STATUS_REG,
				  E1432_STATUS_Q_RESP_READY,
				  E1432_STATUS_Q_RESP_READY,
				  CMD_READY_TIMEOUT,
				  "Q RESP READY bit in STATUS reg");
    if (error < 0)
	return error;

    /* Read response */
    error = i1432_direct_read32_register(mn, E1432_QUERY_RESPONSE_REG,
					 resp);
    if (error < 0)
	return error;

    return 0;
}

SHORTSIZ16
i1432_read_resp4(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 *resp1,
		 LONGSIZ32 *resp2, LONGSIZ32 *resp3, LONGSIZ32 *resp4)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;

    error = i1432_get_module_from_chan(hw, ID, &mn);
    if (error)
	return error;

    /* Wait for Q RESP READY */
    error = i1432_wait_bits_match(mn, E1432_STATUS_REG,
				  E1432_STATUS_Q_RESP_READY,
				  E1432_STATUS_Q_RESP_READY,
				  CMD_READY_TIMEOUT,
				  "Q RESP READY bit in STATUS reg");
    if (error < 0)
	return error;

    /* Read responses */
    error = i1432_direct_read32_register(mn, E1432_QUERY_RESPONSE_REG,
					 resp1);
    if (error < 0)
	return error;
    error = i1432_direct_read32_register(mn, E1432_PARAMETER_1_REG,
					 resp2);
    if (error < 0)
	return error;
    error = i1432_direct_read32_register(mn, E1432_PARAMETER_2_REG,
					 resp3);
    if (error < 0)
	return error;
    error = i1432_direct_read32_register(mn, E1432_PARAMETER_3_REG,
					 resp4);
    if (error < 0)
	return error;

    return 0;
}

static SHORTSIZ16
cmd_setup(E1432ID hw, SHORTSIZ16 ID, E1432_MODULE_LIST_NODE **mn)
{
    SHORTSIZ16 error;

    error = i1432_get_module_from_chan(hw, ID, mn);
    if (error)
	return error;

    /* Wait for CMD READY (and !DONE for software handshake) */
    error = i1432_wait_bits_match(*mn, E1432_STATUS_REG,
				  E1432_STATUS_CMD_READY,
				  E1432_STATUS_CMD_READY,
				  CMD_READY_TIMEOUT,
				  "CMD READY bit in STATUS reg");
    if (error < 0)
	return error;

    return 0;
}

static SHORTSIZ16
cmd_finish(E1432ID hw, SHORTSIZ16 ID,
	    E1432_MODULE_LIST_NODE *mn, LONGSIZ32 cmd)
{
    FLOATSIZ64 timeout;
    SHORTSIZ16 error;

    if (cmd == E1432_CMD_AUTO_ZERO)
	timeout = AUTO_ZERO_TIMEOUT;
    else
	timeout = DONE_TIMEOUT;

    /* Write command */
#ifdef	CMD_REG_WORKAROUND
    error = i1432_direct_write32_register(mn, E1432_ALT_COMMAND_REG, cmd);
    if (error < 0)
	return error;
    error = i1432_direct_write32_register(mn, E1432_HOSTB_CVR_REG,
					  E1432_HOSTB_CVR_VXICMD);
    if (error < 0)
	return error;

    error = i1432_wait32_bits_match(mn, E1432_ALT_COMMAND_REG,
				    -1, 0, timeout,
				    "ALT COMMAND reg");
    if (error < 0)
	return error;
#else
    error = i1432_direct_write32_register(mn, E1432_COMMAND_REG, cmd);
    if (error < 0)
	return error;
#endif

    /* Wait for CMD READY and DONE */
    error = i1432_wait_bits_match(mn, E1432_STATUS_REG,
				  E1432_STATUS_CMD_READY |
				  E1432_STATUS_DONE,
				  E1432_STATUS_CMD_READY |
				  E1432_STATUS_DONE, timeout,
				  "CMD READY and DONE bits in STATUS reg");
    if (error < 0)
	return error;

    /* Clean up */
    if (cmd != E1432_CMD_READ_ERROR)
    {
	error = i1432_error_check(hw, ID);
	if (error < 0)
	    return error;
    }

    return 0;
}

SHORTSIZ16
i1432_write_cmd0(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 cmd)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error2;

    error = i1432_introff();
    if (error)
	return error;

    error = cmd_setup(hw, ID, &mn);
    if (error)
	goto cleanup;

    error = cmd_finish(hw, ID, mn, cmd);
    if (error)
	goto cleanup;

 cleanup:
    error2 = i1432_intron();
    if (error2)
	return error2;

    if (error)
	return error;

    return 0;
}

SHORTSIZ16
i1432_write_cmd1(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 cmd,
		 LONGSIZ32 parm1)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error2;

    error = i1432_introff();
    if (error)
	return error;

    error = cmd_setup(hw, ID, &mn);
    if (error)
	goto cleanup;

    /* Write parm1 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_1_REG,
					  parm1);
    if (error < 0)
	goto cleanup;

    error = cmd_finish(hw, ID, mn, cmd);
    if (error)
	goto cleanup;

 cleanup:
    error2 = i1432_intron();
    if (error2)
	return error2;

    if (error)
	return error;

    return 0;
}

SHORTSIZ16
i1432_write_cmd1_start(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 cmd,
		 LONGSIZ32 parm1)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;
    FLOATSIZ64 timeout;

    if (cmd == E1432_CMD_AUTO_ZERO)
	timeout = AUTO_ZERO_TIMEOUT;
    else
	timeout = DONE_TIMEOUT;

    error = cmd_setup(hw, ID, &mn);
    if (error)
	return error;

    /* Write parm1 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_1_REG,
					  parm1);
    if (error < 0)
	return error;

    /* Write command */
#ifdef	CMD_REG_WORKAROUND
    error = i1432_direct_write32_register(mn, E1432_ALT_COMMAND_REG, cmd);
    if (error < 0)
	return error;
    error = i1432_direct_write32_register(mn, E1432_HOSTB_CVR_REG,
					  E1432_HOSTB_CVR_VXICMD);
    if (error < 0)
	return error;

    error = i1432_wait32_bits_match(mn, E1432_ALT_COMMAND_REG,
				    -1, 0, timeout,
				    "ALT COMMAND reg");
    if (error < 0)
	return error;
#else
    error = i1432_direct_write32_register(mn, E1432_COMMAND_REG, cmd);
    if (error < 0)
	return error;
#endif

    return 0;
}

SHORTSIZ16
i1432_write_cmd2(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 cmd,
		 LONGSIZ32 parm1, LONGSIZ32 parm2)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error2;

    error = i1432_introff();
    if (error)
	return error;

    error = cmd_setup(hw, ID, &mn);
    if (error)
	goto cleanup;

    /* Write parm1 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_1_REG,
					  parm1);
    if (error < 0)
	goto cleanup;

    /* Write parm2 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_2_REG,
					  parm2);
    if (error < 0)
	goto cleanup;

    error = cmd_finish(hw, ID, mn, cmd);
    if (error)
	goto cleanup;

 cleanup:
    error2 = i1432_intron();
    if (error2)
	return error2;

    if (error)
	return error;

    return 0;
}

SHORTSIZ16
i1432_write_cmd3(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 cmd,
		 LONGSIZ32 parm1, LONGSIZ32 parm2, LONGSIZ32 parm3)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error2;

    error = i1432_introff();
    if (error)
	return error;

    error = cmd_setup(hw, ID, &mn);
    if (error)
	goto cleanup;

    /* Write parm1 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_1_REG,
					  parm1);
    if (error < 0)
	goto cleanup;

    /* Write parm2 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_2_REG,
					  parm2);
    if (error < 0)
	goto cleanup;

    /* Write parm3 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_3_REG,
					  parm3);
    if (error < 0)
	goto cleanup;

    error = cmd_finish(hw, ID, mn, cmd);
    if (error)
	goto cleanup;

 cleanup:
    error2 = i1432_intron();
    if (error2)
	return error2;

    if (error)
	return error;

    return 0;
}


SHORTSIZ16
i1432_write_cmd4(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 cmd,
		 LONGSIZ32 parm1, LONGSIZ32 parm2, LONGSIZ32 parm3,
		 LONGSIZ32 parm4)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error2;

    error = i1432_introff();
    if (error)
	return error;

    error = cmd_setup(hw, ID, &mn);
    if (error)
	goto cleanup;

    /* Write parm1 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_1_REG,
					  parm1);
    if (error < 0)
	goto cleanup;

    /* Write parm2 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_2_REG,
					  parm2);
    if (error < 0)
	goto cleanup;

    /* Write parm3 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_3_REG,
					  parm3);
    if (error < 0)
	goto cleanup;

    /* Write parm4 */
    error = i1432_direct_write32_register(mn, E1432_PARAMETER_4_REG,
					  parm4);
    if (error < 0)
	goto cleanup;

    error = cmd_finish(hw, ID, mn, cmd);
    if (error)
	goto cleanup;

 cleanup:
    error2 = i1432_intron();
    if (error2)
	return error2;

    if (error)
	return error;

    return 0;
}

SHORTSIZ16
i1432_write_cmd_finish(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 cmd)
{
    E1432_MODULE_LIST_NODE *mn;
    FLOATSIZ64 timeout;
    SHORTSIZ16 error;

    error = i1432_get_module_from_chan(hw, ID, &mn);
    if (error)
	return error;

    if (cmd == E1432_CMD_AUTO_ZERO)
	timeout = AUTO_ZERO_TIMEOUT;
    else
	timeout = DONE_TIMEOUT;

    /* Wait for CMD READY and DONE */
    error = i1432_wait_bits_match(mn, E1432_STATUS_REG,
				  E1432_STATUS_CMD_READY |
				  E1432_STATUS_DONE,
				  E1432_STATUS_CMD_READY |
				  E1432_STATUS_DONE, timeout,
				  "CMD READY and DONE bits in STATUS reg");
    if (error < 0)
	return error;

    /* Clean up */
    if (cmd != E1432_CMD_READ_ERROR)
    {
	error = i1432_error_check(hw, ID);
	if (error < 0)
	    return error;
    }

    return 0;
}

/*
 *********************************************************************
 Writes a word, <data>, to an individual register, <reg> for the
 channel or group of channels in <ID>.  Returns negative error number
 if error, else returns 0;
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_write_register(E1432ID hw, SHORTSIZ16 chanID, LONGSIZ32 reg,
		     SHORTSIZ16 data)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_write_register(0x%p, %d, %ld, %d)\n",
		     hw, chanID, reg, data));
    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);
    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
	return error;
    return i1432_direct_write_register(mn, reg, data);
}

SHORTSIZ16 EXPORT
e1432_write32_register(E1432ID hw, SHORTSIZ16 chanID, LONGSIZ32 reg,
		       LONGSIZ32 data)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_write32_register(0x%p, %d, %ld, %ld)\n",
		     hw, chanID, reg, data));
    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);
    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
	return error;
    return i1432_direct_write32_register(mn, reg, data);
}

/*
 *********************************************************************
 Reads word into <data> from an individual register, located at <reg>
 from logical channel, <chanID>.  Returns negative error number if
 error, else returns 0;
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_read_register(E1432ID hw, SHORTSIZ16 chanID, LONGSIZ32 reg,
		    SHORTSIZ16 * data)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_read_register(0x%p, %d, %ld, 0x%p)\n",
		     hw, chanID, reg, data));
    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);
    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
	return error;
    return i1432_direct_read_register(mn, reg, data);
}

SHORTSIZ16 EXPORT
e1432_read32_register(E1432ID hw, SHORTSIZ16 chanID, LONGSIZ32 reg,
		      LONGSIZ32 * data)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_read32_register(0x%p, %d, %ld, 0x%p)\n",
		     hw, chanID, reg, data));
    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);
    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
	return error;
    return i1432_direct_read32_register(mn, reg, data);
}

/*
 *********************************************************************
 This routine returns the address of a register of an E1432 channel
 whose ID is <chanID>, and whose register offset is <reg>, into
 <regAddr>.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_get_register_address(E1432ID hw, SHORTSIZ16 chanID,
			   LONGSIZ32 reg,
			   volatile SHORTSIZ16 ** regAddrPtr)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;

    error = i1432_get_module_from_chan(hw, chanID, &mn);
    if (error)
	return error;
    error = i1432_calc_register_address(mn, reg, regAddrPtr);
    if (error)
	return error;
    return 0;
}

/*
 *********************************************************************
 Badly named function.  Controls whether we print something for each
 hardware register access.  A value of zero means no prints.  A value
 of one means print write access only.  A value of two or more means
 print both read and write accesses.
 *********************************************************************
 */
void EXPORT
e1432_debug_level(SHORTSIZ16 value)
{
    TRACE_PRINTF(0, ("e1432_debug_level(%d)\n", value));
    i1432_print_reg_access = value;
}

/*
 *********************************************************************
 Function trace level.
 *********************************************************************
 */
void EXPORT
e1432_trace_level(SHORTSIZ16 level)
{
    TRACE_PRINTF(0, ("e1432_trace_level(%d)\n", level));
    i1432_trace_level = level;
}

#ifdef	HPVXI_DOWNLOAD
static SHORTSIZ16
hpvxi_download_imap(E1432_MODULE_LIST_NODE *mn)
{
    ULONGSIZ32 fifo_base;
    const ULONGSIZ32 page_size = 65536;

    if (mn->a24_base == NULL)
    {
	if (mn->pos_a24_handle == NULL)
	    return i1432_la_print_error(mn->la,
					ERR1432_UNABLE_TO_MAP_MEMORY);
	mn->a24_base = os_a24_on(mn->pos_a24_handle);
	if (mn->a24_base == NULL)
	    return i1432_la_print_error(mn->la,
					ERR1432_UNABLE_TO_MAP_MEMORY);
	if (mn->a24_256k)
	    /* Point a24_base to movable window */
	    mn->a24_base = (LONGSIZ32 *)
		((unsigned long) mn->a24_base + E1432_WINDOW_BASE_256K);
    }

    /* We map in all of the module's A24 space, so just point
       a24_fifo_base to the FIFO portion of it */
    fifo_base = mn->a24_base;
    if (mn->a24_256k)
    {
	/* Point to the fixed window, first */
	fifo_base -= E1432_WINDOW_BASE_256K;
	fifo_base += FIFO_PAGE_START_256K * page_size;
    }
    else
	fifo_base += FIFO_PAGE_START_1M * page_size;
    mn->a24_fifo_base = (LONGSIZ32 *) fifo_base;

    return 0;
}

static SHORTSIZ16
i1432_hpvxi_download_iunmap(E1432_MODULE_LIST_NODE *mn)
{
    /* Must turn off A24 addressing before doing A16 addressing since
       os_a24_on() may unmap A16 space.  See SCPI Toolkit HP VXI
       Function Reference */
    if (mn->a24_base != NULL)
    {
	os_a24_off(mn->pos_a24_handle);
	mn->a24_base = NULL;
	mn->a24_fifo_base = NULL;
    }

    return 0;
}
#endif

SHORTSIZ16
i1432_imap(E1432_MODULE_LIST_NODE *mn, int use_fifo)
{
#ifdef	HAVE_SICL
    SHORTSIZ16 error;

    error = i1432_sicl_imap(mn, use_fifo);
    if (error)
	return error;
#endif
#ifdef	HPVXI_DOWNLOAD
    SHORTSIZ16 error;

    error = hpvxi_download_imap(mn);
    if (error)
	return error;
#endif

    return 0;
}

SHORTSIZ16
i1432_iunmap(E1432_MODULE_LIST_NODE *mn, int use_fifo)
{
#ifdef	HAVE_SICL
    SHORTSIZ16 error;

    error = i1432_sicl_iunmap(mn, use_fifo);
    if (error)
	return error;
#endif
#ifdef	HPVXI_DOWNLOAD
    SHORTSIZ16 error;

    if (!use_fifo)
    {
	error = i1432_hpvxi_download_iunmap(mn);
	if (error)
	    return error;
    }
#endif

    return 0;
}


#ifndef HAVE_VTL /* all I/O functions move to vtlio.c for VTL */

/*
 *********************************************************************
 Returns the address of a register in a particular module.
 *********************************************************************
 */
SHORTSIZ16
i1432_calc_register_address(E1432_MODULE_LIST_NODE *mn,
			    LONGSIZ32 reg,
			    volatile SHORTSIZ16 ** regAddrPtr)
{
    return i1432_calc_register_address_a(mn, reg, (reg >= 0x40), regAddrPtr);
}

SHORTSIZ16
i1432_calc_register_address_a(E1432_MODULE_LIST_NODE *mn,
			      LONGSIZ32 reg, int a24,
			      volatile SHORTSIZ16 ** regAddrPtr)
{
    unsigned long addr;
    SHORTSIZ16 error;

    if (a24)
    {
	error = i1432_imap(mn, 0);
	if (error)
	    return error;
	addr = (unsigned long) mn->a24_base;
    }
    else
    {
#ifdef  HPVXI_DOWNLOAD
	error = i1432_hpvxi_download_iunmap(mn);
	if (error)
	    return error;
#endif
	addr = (unsigned long) mn->a16_base;
    }

    addr += reg;
    *regAddrPtr = (volatile SHORTSIZ16 *) addr;

    TRACE_PRINTF(9, ("calc_register_address(la=%d, 0x%lx) returns 0x%p\n",
		     mn->la, reg, (char *) addr));

    return 0;
}


/*
 *********************************************************************
 Write 16 bit data to an address in a VXI module.  Returns negative
 error number if error, else returns 0.
 *********************************************************************
 */
SHORTSIZ16
i1432_direct_write_register(E1432_MODULE_LIST_NODE *mn,
			    LONGSIZ32 reg, SHORTSIZ16 data)
{
    volatile SHORTSIZ16 *addr;
    SHORTSIZ16 error;
    SHORTSIZ16 rv = 0;
    int     need_intron = 0;

#ifdef	ALLOW_PRINTF
    if (i1432_print_reg_access > 0)
	PRINTF(("WRITE to LA %d, register 0x%lx, value 0x%x\n",
		mn->la, reg, data & 0xffff));
#endif

    /* ----- General setup ----- */
    /* Block interrupts between mapping memory and using the memory */
    error = i1432_introff();
    if (error)
    {
	rv = error;
	goto cleanup;
    }
    need_intron = 1;

    /* ----- Set up bus error checking ----- */
#ifdef	HAVE_SIGBUS
    if (i1432_buserr_trap)
    {
	if (setjmp(i1432_buserr_env) != 0)
	{
	    /* We are recovering from bus fault */
	    rv = i1432_buserr_report(mn->la, reg, "write");
	    goto cleanup;
	}
	i1432_buserr_env_valid = 1;
    }
#endif

#ifdef	HPVXI_DOWNLOAD
    {
#pragma	ASM
	XREF	_os_setup_bus_error
	PEA	on_write_bus_error
	JSR	_os_setup_bus_error
	ADDQ.L	#4,SP
#pragma	END_ASM
    }
    if (0)
    {
#pragma	ASM
on_write_bus_error	EQU	*
#pragma	END_ASM
	/* We are recovering from bus fault */
	rv = i1432_buserr_report(mn->la, reg, "write");
	goto cleanup;
    }
#endif	/* HPVXI_DOWNLOAD */

#ifdef	EMULATE_SICL
    esicl_clear_bus_error();
#endif

    rv = i1432_calc_register_address(mn, reg, &addr);
    if (rv)
	goto cleanup;

    /* This code assumes that the only 16-bit pokes will be to true
       16-bit registers in A16 space, so a simple iwpoke will work. */
    iwpoke((unsigned short *) addr, (unsigned short) data);

 cleanup:
    /* ----- Clean up bus error checking ----- */
#ifdef	HAVE_SIGBUS
#ifdef	_HPUX_SOURCE
    /* Apparently delays CPU pipeline long enough for
       i1432_buserr_env_valid to not be set to 0 before the bus error
       is handled. */
    if (i1432_buserr_env_valid)
#endif
	i1432_buserr_env_valid = 0;
#endif

#ifdef	HPVXI_DOWNLOAD
    (void) os_unsetup_bus_error();
#endif

#ifdef	EMULATE_SICL
    if (esicl_bus_error())
	rv = i1432_buserr_report(mn->la, reg, "write");
#endif

    /* ----- General cleanup ----- */
    if (need_intron)
    {
	error = i1432_intron();
	if (error)
	    rv = error;
    }

    return rv;
}

/*
 *********************************************************************
 Write 32 bit data to reg at la.  Returns negative error number if
 error, else returns 0.
 *********************************************************************
 */
SHORTSIZ16
i1432_direct_write32_register(E1432_MODULE_LIST_NODE *mn,
			      LONGSIZ32 reg, LONGSIZ32 data)
{
    return i1432_direct_write32_register_a(mn, reg, (reg >= 0x40), data);
}

SHORTSIZ16
i1432_direct_write32_register_a(E1432_MODULE_LIST_NODE *mn,
			      LONGSIZ32 reg, int a24, LONGSIZ32 data)
{
    volatile LONGSIZ32 *addr;
    SHORTSIZ16 rv = 0;
    SHORTSIZ16 error;
    int     need_intron = 0;

#ifdef	ALLOW_PRINTF
    if (i1432_print_reg_access > 0)
	PRINTF(("WRITE32 to LA %d, register 0x%lx, value 0x%lx\n",
		mn->la, reg, data));
#endif

    /* ----- General setup ----- */
    /* Block interrupts between mapping memory and using the memory */
    error = i1432_introff();
    if (error)
    {
	rv = error;
	goto cleanup;
    }
    need_intron = 1;

    /* ----- Set up bus error checking ----- */
#ifdef	HAVE_SIGBUS
    if (i1432_buserr_trap)
    {
	if (setjmp(i1432_buserr_env) != 0)
	{
	    /* We are recovering from bus fault */
	    rv = i1432_buserr_report(mn->la, reg, "write32");
	    goto cleanup;
	}
	i1432_buserr_env_valid = 1;
    }
#endif

#ifdef	HPVXI_DOWNLOAD
    {
#pragma	ASM
	XREF	_os_setup_bus_error
	PEA	on_write32_bus_error
	JSR	_os_setup_bus_error
	ADDQ.L	#4,SP
#pragma	END_ASM
    }
    if (0)
    {
#pragma	ASM
on_write32_bus_error	EQU	*
#pragma	END_ASM
	/* We are recovering from bus fault */
	rv = i1432_buserr_report(mn->la, reg, "write32");
	goto cleanup;
    }
#endif	/* HPVXI_DOWNLOAD */

#ifdef	EMULATE_SICL
    esicl_clear_bus_error();
#endif

    rv = i1432_calc_register_address_a(mn, reg, a24,
				       (volatile SHORTSIZ16 **)
				       &addr);
    if (rv)
	goto cleanup;

    if (mn->d32)
	ilpoke((unsigned long *) addr, (unsigned long) data);
    else
    {
	iwpoke((volatile unsigned short *) addr,
	       (unsigned short) (data >> 16));
	/* We shouldn't need the 0xffff on the next line, but without
	   it the optimizer messes up and re-orders the two iwpokes,
	   causing data corruption between the host and Semaphore. */
	iwpoke(((volatile unsigned short *) addr) + 1,
	       (unsigned short) (data & 0xffff));
    }

 cleanup:
    /* ----- Clean up bus error checking ----- */
#ifdef	HAVE_SIGBUS
#ifdef	_HPUX_SOURCE
    /* Apparently delays CPU pipeline long enough for
       i1432_buserr_env_valid to not be set to 0 before the bus error
       is handled. */
    if (i1432_buserr_env_valid)
#endif
	i1432_buserr_env_valid = 0;
#endif

#ifdef	HPVXI_DOWNLOAD
    (void) os_unsetup_bus_error();
#endif

#ifdef	EMULATE_SICL
    if (esicl_bus_error())
	rv = i1432_buserr_report(mn->la, reg, "write32");
#endif

    /* ----- General cleanup ----- */
    if (need_intron)
    {
	error = i1432_intron();
	if (error)
	    rv = error;
    }

    return rv;
}

/*
 *********************************************************************
 Read 16 bit data from reg at la.  Returns negative error number if
 error, else returns 0.
 *********************************************************************
 */
SHORTSIZ16
i1432_direct_read_register(E1432_MODULE_LIST_NODE *mn,
			   LONGSIZ32 reg, SHORTSIZ16 * data)
{
    volatile SHORTSIZ16 *addr;
    SHORTSIZ16 rv = 0;
    SHORTSIZ16 error;
    int     need_intron = 0;

    /* ----- General setup ----- */
    /* Block interrupts between mapping memory and using the memory */
    error = i1432_introff();
    if (error)
    {
	rv = error;
	goto cleanup;
    }
    need_intron = 1;

    /* ----- Set up bus error checking ----- */
#ifdef	HAVE_SIGBUS
    if (i1432_buserr_trap)
    {
	if (setjmp(i1432_buserr_env) != 0)
	{
	    /* We are recovering from a bus fault */
	    rv = i1432_buserr_report(mn->la, reg, "read");
	    goto cleanup;
	}
	i1432_buserr_env_valid = 1;
    }
#endif

#ifdef	HPVXI_DOWNLOAD
    {
#pragma	ASM
	XREF	_os_setup_bus_error
	PEA	on_read_bus_error
	JSR	_os_setup_bus_error
	ADDQ.L	#4,SP
#pragma	END_ASM
    }
    if (0)
    {
#pragma	ASM
on_read_bus_error	EQU	*
#pragma	END_ASM
	/* We are recovering from a bus fault */
	rv = i1432_buserr_report(mn->la, reg, "read");
	goto cleanup;
    }
#endif	/* HPVXI_DOWNLOAD */

#ifdef	EMULATE_SICL
    esicl_clear_bus_error();
#endif

    rv = i1432_calc_register_address(mn, reg, &addr);
    if (rv)
	goto cleanup;

    /* This code assumes that the only 16-bit peeks will be to
       true 16-bit registers in A16 space, so a simple iwpeek
       will work. */
    *data = iwpeek((volatile unsigned short *) addr);

 cleanup:
    /* ----- Clean up bus error checking ----- */
#ifdef	HAVE_SIGBUS
#ifdef	_HPUX_SOURCE
    /* Apparently delays CPU pipeline long enough for
       i1432_buserr_env_valid to not be set to 0 before the bus error
       is handled. */
    if (i1432_buserr_env_valid)
#endif
	i1432_buserr_env_valid = 0;
#endif

#ifdef	HPVXI_DOWNLOAD
    (void) os_unsetup_bus_error();
#endif

#ifdef	EMULATE_SICL
    if (esicl_bus_error())
	rv = i1432_buserr_report(mn->la, reg, "read");
#endif

    /* ----- General cleanup ----- */
    if (need_intron)
    {
	error = i1432_intron();
	if (error)
	    rv = error;
    }

#ifdef	ALLOW_PRINTF
    if (rv == 0 && i1432_print_reg_access > 1)
	PRINTF(("READ from LA %d, register 0x%lx, value 0x%x\n",
		mn->la, reg, *data & 0xffff));
#endif

    return rv;
}

/*
 *********************************************************************
 Read 32 bit data from reg at la.  Returns negative error number if
 error, else returns 0.
 *********************************************************************
 */
SHORTSIZ16
i1432_direct_read32_register(E1432_MODULE_LIST_NODE *mn,
			     LONGSIZ32 reg, LONGSIZ32 * data)
{
    return i1432_direct_read32_register_a(mn, reg, (reg >= 0x40), data);
}

SHORTSIZ16
i1432_direct_read32_register_a(E1432_MODULE_LIST_NODE *mn,
			       LONGSIZ32 reg, int a24, LONGSIZ32 * data)
{
    volatile LONGSIZ32 *addr;
    SHORTSIZ16 rv = 0;
    SHORTSIZ16 error;
    int     need_intron = 0;

    /* ----- General setup ----- */
    /* Block interrupts between mapping memory and using the memory */
    error = i1432_introff();
    if (error)
    {
	rv = error;
	goto cleanup;
    }
    need_intron = 1;

    /* ----- Set up bus error checking ----- */
#ifdef	HAVE_SIGBUS
    if (i1432_buserr_trap)
    {
	if (setjmp(i1432_buserr_env) != 0)
	{
	    /* We are recovering from bus fault */
	    rv = i1432_buserr_report(mn->la, reg, "read32");
	    goto cleanup;
	}
	i1432_buserr_env_valid = 1;
    }
#endif

#ifdef	HPVXI_DOWNLOAD
    {
#pragma	ASM
	XREF	_os_setup_bus_error
	PEA	on_read32_bus_error
	JSR	_os_setup_bus_error
	ADDQ.L	#4,SP
#pragma	END_ASM
    }
    if (0)
    {
#pragma	ASM
on_read32_bus_error	EQU	*
#pragma	END_ASM
	/* We are recovering from bus fault */
	rv = i1432_buserr_report(mn->la, reg, "read32");
	goto cleanup;
    }
#endif	/* HPVXI_DOWNLOAD */

#ifdef	EMULATE_SICL
    esicl_clear_bus_error();
#endif

    rv = i1432_calc_register_address_a(mn, reg, a24,
				       (volatile SHORTSIZ16 **)
				       &addr);
    if (rv)
	goto cleanup;

    if (mn->d32)
	/* One 32-bit read */
	*data = ilpeek((unsigned long *) addr);
    else
    {
	/* Combine two 16-bit reads */
	*data = (LONGSIZ32) iwpeek((unsigned short *) addr) << 16;
	*data |= iwpeek(((unsigned short *) addr) + 1);
    }

 cleanup:
    /* ----- Clean up bus error checking ----- */
#ifdef	HAVE_SIGBUS
#ifdef	_HPUX_SOURCE
    /* Apparently delays CPU pipeline long enough for
       i1432_buserr_env_valid to not be set to 0 before the bus error
       is handled. */
    if (i1432_buserr_env_valid)
#endif
	i1432_buserr_env_valid = 0;
#endif

#ifdef	HPVXI_DOWNLOAD
    (void) os_unsetup_bus_error();
#endif

#ifdef	EMULATE_SICL
    if (esicl_bus_error())
	rv = i1432_buserr_report(mn->la, reg, "read32");
#endif

    /* ----- General cleanup ----- */
    if (need_intron)
    {
	error = i1432_intron();
	if (error)
	    rv = error;
    }

#ifdef	ALLOW_PRINTF
    if (rv == 0 && i1432_print_reg_access > 1)
	PRINTF(("READ32 from LA %d, register 0x%lx, value 0x%x\n",
		mn->la, reg, *data));
#endif

    return rv;
}

#endif /* HAVE_VTL */

#ifdef	HAVE_SIGBUS
/*ARGSUSED*/
static void
tmp_buserr_handler(LONGSIZ32 sig, LONGSIZ32 code,
		   struct sigcontext *scp)
{
    /* return to previous context, signal = 1 */
    longjmp(i1432_buserr_env, 1);
}
#endif

/*
 *********************************************************************
 This is much like i1432_direct_read_register.  The differences: we
 think there is a reasonable chance of a bus error, so we always trap
 bus errors if the host supports it, and we allow much longer before
 undoing our handler and restoring the default.  This works around a
 problem on s700 where the bus error happens some time after the
 transfer took place.  We don't care about the value read, we just
 want to know if the read worked, so we don't return the value.

 This could be #ifndef HAVE_VTL, except that it is called from
 testsca.c.
 *********************************************************************
 */
SHORTSIZ16
i1432_direct_ping_register(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 reg)
{
    SHORTSIZ16 dummy;
#ifdef	HAVE_SIGBUS
    struct sigvec vec;
    SHORTSIZ16 error;
    int     save_state;

    save_state = i1432_buserr_trap;
    i1432_buserr_trap = 0;	/* Don't use normal set-try-recover */

    /* Set up our own buserr handler */
    vec.sv_handler = tmp_buserr_handler;
    vec.sv_mask = 0;
#ifdef E1485_SOURCE
    vec.sv_flags = SV_FLAGS_DEFAULT;
#else
    vec.sv_flags = 0;
#endif
    (void) sigvector(SIGBUS, &vec, &vec);

    if (setjmp(i1432_buserr_env) == 0)
    {
	/* Do the I/O, and wait in case a bus error is delayed */
	error = i1432_direct_read_register(mn, reg, &dummy);
	i1432_pause(0.005);
    }
    else
	error = i1432_buserr_report(mn->la, reg, "ping");

    /* Restore old handler and set-try-recover state */
    (void) sigvector(SIGBUS, &vec, &vec);
    i1432_buserr_trap = save_state;
    return error;
#else
    return i1432_direct_read_register(mn, reg, &dummy);
#endif
}

#if defined(HAVE_SICL) || defined(HPVXI_DOWNLOAD)
/*
 *********************************************************************
 Yes, believe it or not this is significantly faster than going
 through ilblockcopy or ilpopfifo on a s700 running HP-UX 9.05 with
 sicl C_03_08b.  Even a non-unrolled loop of ilpeek/ilpokes is faster.
 Stupid.

 HPVXI_DOWNLOAD does not even have ilblockcopy, so it must always use
 the unrolled version.
 *********************************************************************
 */
static void
unrolled_lblockcopy(unsigned long *src, unsigned long *dest,
		    unsigned long cnt)
{
    long    i;
    unsigned long val1, val2, val3, val4;
    unsigned long val5, val6, val7, val8;

    for (i = 0; i < (long) cnt % 8; i++)
	*dest++ = ilpeek(src++);
    cnt /= 8;
    for (i = 0; i < (long) cnt; i++)
    {
	val1 = ilpeek(src++);
	val2 = ilpeek(src++);
	val3 = ilpeek(src++);
	val4 = ilpeek(src++);
	val5 = ilpeek(src++);
	val6 = ilpeek(src++);
	val7 = ilpeek(src++);
	val8 = ilpeek(src++);
	/* For speed on V743, these should all be together after the
	   peeks above. */
	*dest++ = val1;
	*dest++ = val2;
	*dest++ = val3;
	*dest++ = val4;
	*dest++ = val5;
	*dest++ = val6;
	*dest++ = val7;
	*dest++ = val8;
    }
}

static void
unrolled_wblockcopy(unsigned short *src, unsigned short *dest,
		    unsigned long cnt)
{
    long    i;
    unsigned short val1, val2, val3, val4;
    unsigned short val5, val6, val7, val8;

    for (i = 0; i < (long) cnt % 8; i++)
	*dest++ = iwpeek(src++);
    cnt /= 8;
    for (i = 0; i < (long) cnt; i++)
    {
	val1 = iwpeek(src++);
	val2 = iwpeek(src++);
	val3 = iwpeek(src++);
	val4 = iwpeek(src++);
	val5 = iwpeek(src++);
	val6 = iwpeek(src++);
	val7 = iwpeek(src++);
	val8 = iwpeek(src++);
	*dest++ = val1;
	*dest++ = val2;
	*dest++ = val3;
	*dest++ = val4;
	*dest++ = val5;
	*dest++ = val6;
	*dest++ = val7;
	*dest++ = val8;
    }
}
#endif

static SHORTSIZ16
xfer_buserr(E1432_MODULE_LIST_NODE *mn,
	    unsigned long *src, unsigned long *dest, int isread)
{
    long offset;

    if (isread)
	offset = (long) src;
    else
	offset = (long) dest;

    return i1432_buserr_report(mn->la, offset, "i1432_xfer_block");
}

/* This assumes the xfer uses A24 space.  The count is in terms of
   32-bit words.  If the I/O path can do D32 transfers, and they are
   not currently disabled due to local bus, this function will try to
   use D32 transfers.  Otherwise it will fall back to D16
   transfers.  This function attempts to trap bus errors if
   possible, though I'll consider removing that at some point to make
   things more efficient.

   This function does not deal with swapping, so the caller must deal
   with it.

   For reads, the src address is the OFFSET from the a24_base or
   a24_fifo_base.  For writes, the dest address is the OFFSET fromthe
   a24_base or a24_fifo_base.  The caller can't add in a24_base or
   a24_fifo_base, because these values might not be correct until
   after we map in the memory inside this function. */
SHORTSIZ16
i1432_xfer_block(E1432_MODULE_LIST_NODE *mn,
		 unsigned long *src, unsigned long *dest,
		 unsigned long cnt, int use_fifo, int isread)
{
    SHORTSIZ16 rv = 0;
    SHORTSIZ16 error;
    int     need_intron = 0;
#ifndef	HPVXI_DOWNLOAD
    int     ierror;
#endif
#ifdef	HAVE_VTL
SHORTSIZ16 	err;
#endif

#ifdef	DEBUG_XFER
    (void) printf("i1432_xfer_block(%d, 0x%p, 0x%p, %lu, %d, %d)\n",
		  mn - i1432_mod_list, src, dest, cnt,
		  use_fifo, isread);
#endif

    /* ----- General setup ----- */
    /* Block interrupts between mapping memory and using the memory */
    error = i1432_introff();
    if (error)
    {
	rv = error;
	goto cleanup;
    }
    need_intron = 1;
#ifdef	HAVE_VTL
    err=i1432_module_connect(mn);
    if(err)
	return(err);
#endif
    /* Make sure we have mapped in the appropriate memory */
    rv = i1432_imap(mn, use_fifo);
    if (rv)
	goto cleanup;

    /* ----- Set up bus error checking ----- */
#ifdef	HAVE_SIGBUS
    if (i1432_buserr_trap)
    {
	if (setjmp(i1432_buserr_env) != 0)
	{
	    /* We are recovering from bus fault */
	    rv = xfer_buserr(mn, src, dest, isread);
	    goto cleanup;
	}
	i1432_buserr_env_valid = 1;
    }
#endif

#ifdef	HPVXI_DOWNLOAD
    {
#pragma	ASM
	XREF	_os_setup_bus_error
	PEA	on_xfer_bus_error
	JSR	_os_setup_bus_error
	ADDQ.L	#4,SP
#pragma	END_ASM
    }
    if (0)
    {
#pragma	ASM
on_xfer_bus_error	EQU	*
#pragma	END_ASM
	/* We are recovering from bus fault */
	rv = xfer_buserr(mn, src, dest, isread);
	goto cleanup;
    }
#endif	/* HPVXI_DOWNLOAD */

#ifdef	EMULATE_SICL
    esicl_clear_bus_error();
#endif

    /* ----- Calculate address ----- */
    if (isread)
	if (use_fifo)
	    src = (unsigned long *) ((long) mn->a24_fifo_base + (long) src);
	else
	    src = (unsigned long *) ((long) mn->a24_base + (long) src);
    else
	if (use_fifo)
	    dest = (unsigned long *) ((long) mn->a24_fifo_base + (long) dest);
	else
	    dest = (unsigned long *) ((long) mn->a24_base + (long) dest);

    /* ----- Do the transfer ----- */
#ifdef	HAVE_SICL
#if defined(__hpux) && !SWAP
    if (isread)
    {
	/* Use faster unrolled transfer if not swapping */
	if (mn->d32)
	    unrolled_lblockcopy(src, dest, cnt);
	else
	    unrolled_wblockcopy((unsigned short *) src,
				(unsigned short *) dest, 2 * cnt);
    }
    else
#endif
    {
	if (mn->d32)
	    ierror = ilblockcopy(mn->sicl_id, src, dest, cnt, SWAP);
	else
	    ierror = iwblockcopy(mn->sicl_id, (unsigned short *) src,
				 (unsigned short *) dest, 2 * cnt, SWAP);
	if (ierror != 0)
	{
	    i1432_error_info = igeterrstr(igeterrno());
	    rv = i1432_la_print_error(mn->la, ERR1432_IO);
	    goto cleanup;
	}
    }
#endif

#ifdef	HAVE_VTL
    if (isread)
	if (mn->d32)
	{
	    ierror = ilblockcopy(mn->sicl_id, src, dest, cnt, SWAP);
	    /* If VISA had to back off to 16-bit transfers, clear the
	       module's D32 flag.  It would have been nice if this
	       took place internal to vtlio.c, but the mn pointer is
	       not passed with ilblockcopy. */
	    if (i1432_visa_d16)
		mn->d32 = 0;
	}
	else
	    ierror = iwblockcopy(mn->sicl_id, (unsigned short *) src,
				 (unsigned short *) dest, 2 * cnt, SWAP);
    else
	if (mn->d32)
	{
	    ierror = ilblockcopyout(mn->sicl_id, src, dest, cnt, SWAP);
	    /* If VISA had to back off to 16-bit transfers, clear the
	       module's D32 flag.  It would have been nice if this
	       took place internal to vtlio.c, but the mn pointer is
	       not passed with ilblockcopyout. */
	    if (i1432_visa_d16)
		mn->d32 = 0;
	}
	else
	    ierror = iwblockcopyout(mn->sicl_id, (unsigned short *) src,
				    (unsigned short *) dest, 2 * cnt, SWAP);
    if (ierror != 0)
    {
	rv = i1432_la_print_error(mn->la, ERR1432_IO);
	goto cleanup;
    }
#endif

#ifdef	HPVXI_DOWNLOAD
    /* No i?blockcopy available, so always use unrolled versions.
       This assumes no swap. */
    if (mn->d32)
	unrolled_lblockcopy(src, dest, cnt);
    else
	unrolled_wblockcopy((unsigned short *) src,
			    (unsigned short *) dest, 2 * cnt);
#endif

 cleanup:
    /* ----- Clean up bus error checking ----- */
#ifdef	HAVE_SIGBUS
    i1432_buserr_env_valid = 0;
#endif

#ifdef	HPVXI_DOWNLOAD
    (void) os_unsetup_bus_error();
#endif

#ifdef	EMULATE_SICL
    if (esicl_bus_error())
	rv = xfer_buserr(mn, src, dest, isread);
#endif

    /* ----- General cleanup ----- */
    if (need_intron)
    {
	error = i1432_intron();
	if (error)
	    rv = error;
    }

    return rv;
}

SHORTSIZ16
i1432_data_handshake(E1432_MODULE_LIST_NODE *mn)
{
#ifndef HAVE_VTL
    SHORTSIZ16 rv = 0;
    SHORTSIZ16 error;
    int     need_intron = 0;
    volatile unsigned short *addr;
    FLOATSIZ64 start, current;

#ifdef	DEBUG_XFER
    (void) printf("i1432_data_handshake(%d)\n", mn - i1432_mod_list);
#endif

    /* ----- General setup ----- */
    /* Block interrupts between mapping memory and using the memory */
    error = i1432_introff();
    if (error)
    {
	rv = error;
	goto cleanup;
    }
    need_intron = 1;

    rv = i1432_calc_register_address(mn, E1432_DATA_AVAIL_REG, 
				     (volatile SHORTSIZ16 **) &addr);
    if (rv)
	goto cleanup;

    /* Avoid i1432_get_time overhead if data already ready */
    if (iwpeek(addr) == 0)
    {
	i1432_get_time(&start);
	while (iwpeek(addr) == 0)
	{
	    i1432_get_time(&current);
	    if (current - start > 1 && iwpeek(addr) == 0)
	    {
		rv = i1432_la_print_error(mn->la,
					  ERR1432_DATA_HANDSHAKE_TIMEOUT);
		goto cleanup;
	    }
	}
    }

    iwpoke(addr, 0);
    iwpoke(addr + 1, 0);

 cleanup:

    /* ----- General cleanup ----- */
    if (need_intron)
    {
	error = i1432_intron();
	if (error)
	    rv = error;
    }

    return rv;
#else /* VTL */
    SHORTSIZ16 data, error;
    FLOATSIZ64 start, current;

#ifdef	DEBUG_XFER
    (void) printf("i1432_data_handshake(%d)\n", mn - i1432_mod_list);
#endif

    /* Avoid i1432_get_time overhead if data already ready */
    error = i1432_direct_read_register(mn, E1432_DATA_AVAIL_REG, &data);
    if (error)
	return error;
    if (data == 0)
    {
	i1432_get_time(&start);
	while (data == 0)
	{
	    i1432_get_time(&current);
	    if (current - start > 1)
	    {
		error = i1432_direct_read_register(mn,
						   E1432_DATA_AVAIL_REG,
						   &data);
		if (error)
		    return error;
		if (data == 0)
		    return i1432_la_print_error
			(mn->la, ERR1432_DATA_HANDSHAKE_TIMEOUT);
	    }
	    error = i1432_direct_read_register(mn,
					       E1432_DATA_AVAIL_REG,
					       &data);
	    if (error)
		return error;
	}
    }

    error = i1432_direct_write32_register(mn, E1432_DATA_AVAIL_REG, 0);
    if (error)
	return error;

    return 0;
#endif
}

#if SWAP
/*
 *********************************************************************
 Do 2-byte swap, then 4-byte swap, to fix 4-byte data that has been
 swapped as 2-byte data during transfer.  Used only when SWAP is
 defined.  The nbyte should be a multiple of four; if not, the
 trailing extra bytes are ignored and not swapped in any way.
 *********************************************************************
 */
/*ARGSUSED*/
SHORTSIZ16
i1432_two_four_swap(char *buf, long nbyte)
{
#if 1
    /* This scans the data only once, so should be more efficient than
       two calls to ibeswap.  By using SHORTSIZ16, it is probably
       faster than the implementation that uses chars.  However, it
       assumes that SHORTSIZ16 is exactly two bytes long (which is
       true on all host platforms I know of). */

    SHORTSIZ16 *sbuf = (SHORTSIZ16 *) buf;
    SHORTSIZ16 stmp;
    long    i;
    
    for (i = 0; i < nbyte / 4; i++)
    {
	stmp = *sbuf;
	*sbuf = *(sbuf + 1);
	*sbuf++;
	*sbuf++ = stmp;
    }
#endif
#if 0
    /* This scans the data only once, so should be more efficient than
       two calls to ibeswap.  It's probably slower than the SHORTSIZ16
       loop above, but is completely portable. */

    char    tmp;
    long    i;

    for (i = 0; i < nbyte / 4; i++)
    {
	tmp = *buf;
	*buf = *(buf + 2);
	*(buf + 2) = tmp;

	buf++;

	tmp = *buf;
	*buf = *(buf + 2);
	*(buf + 2) = tmp;

	buf += 3;
    }
#endif
#if 0
    /* Implementation using SICL's ibeswap function. */
    int error;

    /* Undo 2-byte swap */
    error = ibeswap(buf, nbyte, 2);
    if (error != 0)
	return i1432_print_error(ERR1432_IO);

    /* Do 4-byte swap */
    error = ibeswap(buf, nbyte, 4);
    if (error != 0)
	return i1432_print_error(ERR1432_IO);
#endif

    return 0;
}
#endif
